<?php

require_once 'vendor/autoload.php';
use Ramsey\Uuid\Uuid;

require_once (__DIR__ . '/omeka/omekaAPIWrapper.php');
require_once (__DIR__ . '/CameraRawPreviews.php');

/**
 *
 */
define ('INFO_FIELDS_ARRAY', include(__DIR__ . '/fields/infoFields.php'));
/**
 *
 */
define ('TC_FIELDS_ARRAY', include(__DIR__ . '/fields/tcFields.php'));

/**
 * Class TMBProcessor
 */
class TMBProcessor
{

    /** @var omekaAPI wrapper for omeka API */
    private $omekaAPI = null;

    /** @var syncOmeka variable that set if TMBProcessor also sync omeka portal */
    private $syncOmeka = false;

    /** @var php_verbose_mode variable that set if script is in verbose mode */
    private $php_verbose_mode = false;

    /**
     * @var int
     */
    private $currentElaboratedItems = 0;

    /**
     * @var int
     */
    private $elaboratedItems = 0;

    /**
     * @var int
     */
    private $elaboratedCollections = 0;

    /**
     * @var int
     */
    private $maxElaboratedItemsForSendingEmail = 5000;

    /**
     * @var int
     */
    private $timeTMBProcessorStarted = 0;

    /**
     * @return int
     */
    public function getCurrentElaboratedItems(): int
    {
        return $this->currentElaboratedItems;
    }

    /**
     * @param int $currentElaboratedItems
     */
    public function setCurrentElaboratedItems(int $currentElaboratedItems)
    {
        $this->currentElaboratedItems = $currentElaboratedItems;
    }

    /**
     * @return int
     */
    public function getElaboratedItems(): int
    {
        return $this->elaboratedItems;
    }

    /**
     * @param int $elaboratedItems
     */
    public function setElaboratedItems(int $elaboratedItems)
    {
        $this->elaboratedItems = $elaboratedItems;
    }

    /**
     * @return int
     */
    public function getElaboratedCollections(): int
    {
        return $this->elaboratedCollections;
    }

    /**
     * @param int $elaboratedCollections
     */
    public function setElaboratedCollections(int $elaboratedCollections)
    {
        $this->elaboratedCollections = $elaboratedCollections;
    }

    /**
     * @return int
     */
    public function getMaxElaboratedItemsForSendingEmail(): int
    {
        return $this->maxElaboratedItemsForSendingEmail;
    }

    /**
     * @return int
     */
    public function getTimeTMBProcessorStarted(): int
    {
        return $this->timeTMBProcessorStarted;
    }

    /**
     * @param int $timeTMBProcessorStarted
     */
    public function setTimeTMBProcessorStarted(int $timeTMBProcessorStarted)
    {
        $this->timeTMBProcessorStarted = $timeTMBProcessorStarted;
    }

    /**
     * Constructor
     */
    public function __construct($syncOmeka = false, $php_verbose_mode = false)
    {
        $this->syncOmeka = $syncOmeka;
        if ($syncOmeka) {
            $this->omekaAPI = new OmekaAPIWrapper();
            $this->omekaAPI->setVerboseMode(true);
        }
        $this->php_verbose_mode = $php_verbose_mode;
    }

    /**
     * ==================================================
     * imageProcessor
     * ==================================================
     *
     * Method used to process 'TIF' files and create compressed 'JPG'
     *
     * @param object $opObject
     * @param string $source
     * @param string $destination
     * @param boolean $dryMode
     * @return stdClass
     */
    public function imageProcessor(
        $opObject,
        $source,
        $destination,
        $dryMode = false
    )
    {

        $returnObj = new stdClass();
        $returnObj->returnStatus = 'yes';
        $returnObj->returnDescription = '';

        // Check if source is a folder or if not exists
        if (!is_dir($source) && file_exists($source)) {

            try {

                $fileExtension = strtolower(pathinfo($source, PATHINFO_EXTENSION));

                // Checking if the image file is supported for conversion
                if (!in_array($fileExtension, SUPPORTED_IMG_FILE_TYPES)) {
                    if ($this->php_verbose_mode) {
                        error_log("Extension " . $fileExtension . " not supported, skip file...");
                    }
                    $returnObj->returnStatus = 'no';
                    $returnObj->returnDescription = 'Extension ' . $fileExtension . ' not supported!';
                    return $returnObj;
                }

                if (!$dryMode) {
                    if ($this->php_verbose_mode) {
                        error_log("New file to save: " . $destination);
                    }

                    if (strtolower(pathinfo($source, PATHINFO_EXTENSION)) === 'cr2') {

                        /* CameraRawPreviews::extractPreview($source, $destination); */
                        $shellExecCommand = 'ufraw-batch "' . $source . '" --overwrite --compression=100 --exposure=auto --rotate=no --out-type=jpg --output="' . $destination . '"';

                    } else {

                        // Building shell_exec Command
                        $shellExecCommand = 'convert -quiet -quality ' . IMAGE_COMPRESSION_QUALITY . ' "' . $source . '" "' . $destination . '"';

                    }

                    if ($this->php_verbose_mode) {
                        error_log("Command for image conversion: " . $shellExecCommand);
                    }

                    // Write image to system
                    if (shell_exec($shellExecCommand) != null) {
                        $msg = "!ERROR! Impossible to save compressed image";
                        echo $msg;
                        $returnObj->returnStatus = 'no';
                        $returnObj->returnDescription = $msg;
                        return $returnObj;
                    }

                    if ($opObject !== null) $opObject->wsLogMessage('updateInfoWaitingWindow:' . 'Retrieving width and height of image ' . $source . '...');

                    // Retrieve image Width and Height
                    $imageIdentify = shell_exec('identify -format "%wx%h" "' . $destination . '"');
                    // $imageIdentify[0] == ImageWidth && $imageIdentify[1] == ImageHeight
                    $imageIdentify = explode('x', $imageIdentify);
                    if ($this->php_verbose_mode) {
                        error_log("imageIdentify: " . print_r($imageIdentify, true));
                    }

                    if (strtolower(pathinfo($source, PATHINFO_EXTENSION)) !== 'cr2') {
                        if ($opObject !== null) $opObject->wsLogMessage('updateInfoWaitingWindow:' . 'Retrieving exif data of image ' . $source . '...');

                        $shellExecCommand = 'identify -format "%[EXIF:*]" "' . $source . '"';
                        if ($this->php_verbose_mode) {
                            error_log("Command for get image exif data: " . $shellExecCommand);
                        }
                        $exifData = shell_exec($shellExecCommand);

                        // Elimino la dicitura exif: e sostituisco "\n" con ";"
                        $exifData = str_replace('exif:', "", $exifData);
                        $exifData = trim(preg_replace('/\s+/', ';', $exifData));
                        // Togli l'ultimo carattere
                        $exifData = substr($exifData, 0, -1);

                        if ($this->php_verbose_mode) {
                            error_log("exifData: " . print_r($exifData, true));
                        }
                    }
                }

                $returnObj->returnStatus = 'yes';
                $returnObj->returnDescription = 'Image converted with success!';
                if (!$dryMode) {
                    $returnObj->exifData = $exifData;
                    $returnObj->width = intval($imageIdentify[0]);
                    $returnObj->height = intval($imageIdentify[1]);
                } else {
                    $returnObj->exifData = "";
                    $returnObj->width = 0;
                    $returnObj->height = 0;
                }
                return $returnObj;

            } catch (Exception $e) {

                echo $e->getMessage() . "\n";
                $returnObj->returnStatus = 'no';
                $returnObj->returnDescription = $e->getMessage();
                return $returnObj;

            }

        } else {

            $returnObj->returnStatus = 'no';
            $returnObj->returnDescription = 'File ' . $source . ' is a folder or not exist!';
            return $returnObj;

        }

    }

    /**
     * ==================================================
     * infoProcessor
     * ==================================================
     *
     * Method used to process and create info.json
     *
     * @param object $opObject
     * @param string $operation_uuid
     * @param string $inputMetaDataInfoFilePath
     * @param string $outputMetadataInfoFilePath
     * @param string $inputOmekaInfoPath
     * @param boolean $dryMode
     * @param boolean $createPublicCollection
     * @param boolean $importFilze
     * @param object $infoObject
     *
     * @return object|null omeka collection created
     */
    public function infoProcessor(
        $opObject, $operation_uuid,
        $inputMetaDataInfoFilePath = '', $outputMetadataInfoFilePath = '', $inputOmekaInfoPath = '',
        $dryMode = false, $createPublicCollection = false, $importFilze = false,
        $infoObject = null
    )
    {

        if (isset($infoObject)) {

            $objectJSON = $infoObject;

        } else {

            $excelReader = 'Excel2007';
            if (file_exists($inputMetaDataInfoFilePath . '.xlsx')) {
                $inputMetaDataInfoFilePath = $inputMetaDataInfoFilePath . '.xlsx';
            } else if (file_exists($inputMetaDataInfoFilePath . '.xls')) {
                $inputMetaDataInfoFilePath = $inputMetaDataInfoFilePath . '.xls';
                $excelReader = 'Excel5';
            }

            if ($this->php_verbose_mode) {
                error_log("infoProcessor TMBProcessor method");
                error_log("File to load: " . $inputMetaDataInfoFilePath);
            }

            try {

                $cacheMethod = PHPExcel_CachedObjectStorageFactory:: cache_to_phpTemp;
                $cacheSettings = array(' memoryCacheSize ' => '8MB');
                PHPExcel_Settings::setCacheStorageMethod($cacheMethod, $cacheSettings);

                $objReader = PHPExcel_IOFactory::createReader($excelReader);
                $objReader->setReadDataOnly(true);

                if (INFO_METADATAWORKSHEET !== '') {
                    $objReader->setLoadSheetsOnly(array(INFO_METADATAWORKSHEET));
                }

                $objectPHPExcel = $objReader->load($inputMetaDataInfoFilePath);

                if ($this->php_verbose_mode) {
                    error_log("Building INFO PHPExcel object: OK");
                }

                $arrayPHPExcel = $objectPHPExcel->getActiveSheet()->toArray();

                if ($this->php_verbose_mode) {
                    error_log("Turning PHPExcel object into array: OK");
                }

                if ($this->php_verbose_mode) {
                    error_log("HighestColumn: " . $objectPHPExcel->getActiveSheet()->getHighestColumn());
                }

                if ($this->php_verbose_mode) {
                    error_log("HighestRow: " . $objectPHPExcel->getActiveSheet()->getHighestRow());
                }

                // Use array filter to delete blank columns
                $arrayPHPExcel = array_filter(array_map('array_filter', $arrayPHPExcel));
                if ($this->php_verbose_mode) {
                    error_log("Use array filter to delete blank columns: OK");
                }

                $objectPHPExcel->disconnectWorksheets();
                unset($objectPHPExcel);

            } catch (Exception $e) {

                echo "(!)Excel Processor Error: " . $e->getMessage() . "\n";
                if ($this->php_verbose_mode) {
                    error_log("(!)Excel Processor Error: " . $e->getMessage());
                }

            }

            // Building template JSON stdClass object
            $objectJSON = new stdClass();
            $objectJSON->metadata = new stdClass();

            // ===== MAPPING - START =====
            $objectJSON->{'id'} = Uuid::uuid4()->toString();

            if ($this->php_verbose_mode) {
                error_log("info fields array: " . print_r(INFO_FIELDS_ARRAY, true));
            }

            for ($idx = 0; $idx < count($arrayPHPExcel[0]); $idx++) {

                $titleColumn = trim($arrayPHPExcel[0][$idx]);
                if ($this->php_verbose_mode) {
                    error_log("titleColumn: " . $titleColumn);
                }
                if (isset($arrayPHPExcel[1][$idx]) && trim($arrayPHPExcel[1][$idx]) !== '') {
                    $dataColumn = $arrayPHPExcel[1][$idx];
                } else {
                    $dataColumn = "";
                }

                if (in_array($titleColumn, INFO_FIELDS_ARRAY)) {
                    $objectJSON->metadata->{$titleColumn} = $dataColumn;
                } else {
                    echo "\t!INDEX: \"" . $titleColumn . "\" NOT SUPPORTED!\n";
                }

            }

            // Set default values for "Filze" import
            if ($importFilze) {
                if ((!isset($objectJSON->metadata->maintenanceagency)) || (trim($objectJSON->metadata->maintenanceagency) === '')) {
                    $objectJSON->metadata->maintenanceagency = 'SIASVe';
                }
                if ((!isset($objectJSON->metadata->recordid)) || (trim($objectJSON->metadata->recordid) === '')) {
                    $objectJSON->metadata->recordid = 'IT ASVe 0445 010 001 001';
                }
                if ((!isset($objectJSON->metadata->series)) || (trim($objectJSON->metadata->series) === '')) {
                    $objectJSON->metadata->series = 'Redecima 1740 Condizioni di decima. Filze';
                }
                if ((!isset($objectJSON->metadata->level)) || (trim($objectJSON->metadata->level) === '')) {
                    $objectJSON->metadata->level = 'UA';
                }
                if ((!isset($objectJSON->metadata->physdesc)) || (trim($objectJSON->metadata->physdesc) === '')) {
                    $objectJSON->metadata->physdesc = 'carta';
                }
                if ((!isset($objectJSON->metadata->creator)) || (trim($objectJSON->metadata->creator) === '')) {
                    $objectJSON->metadata->creator = 'Dieci savi alle decime in Rialto';
                }
                if ((!isset($objectJSON->metadata->date_year_start)) || (trim($objectJSON->metadata->date_year_start) === '')) {
                    $objectJSON->metadata->date_year_start = '1740';
                }
            }

        }

        if ($this->php_verbose_mode) {
            error_log("JSON Collection loaded:");
            error_log(print_r($objectJSON, true));
        }
        // ===== MAPPING - END =====

        try {

            if (!$dryMode) {
                // Opening in write mode output file pointer
                $filePointer = fopen($outputMetadataInfoFilePath, 'w');
                // Encode JSON into file
                fwrite($filePointer, json_encode($objectJSON));
            }

            $logMessage = "File " . $outputMetadataInfoFilePath . " created!";
            if ($this->php_verbose_mode) {
                error_log($logMessage);
            }

            if ($opObject !== null) $opObject->wsLogMessage('<font color="#006400">' . $logMessage . '</font>');

            $record = new Log(null);
            $record->operation_uuid = $operation_uuid;
            $record->operation = 'Import';
            $record->description = $logMessage;

            $resultInfo = callApi_createLog($record);
            if (!strpos($resultInfo, 'cURL Error')) {
                /*$resultInfo = json_decode($resultInfo);
                $Server->wsSend($clientID, $resultInfo->message);*/
            } else {
                if ($opObject !== null) $opObject->wsLogMessage($resultInfo);
            }

            if (!$dryMode) {
                // Closing previously opened file pointer
                fclose($filePointer);
            }

        } catch (Exception $e) {
            $logMessage = $e->getMessage();

            if ($opObject !== null) $opObject->wsLogMessage('<font color="#8b0000">' . $logMessage . '</font>');

            if ($opObject !== null) $opObject->wsUpdateStatistics($operation_uuid, "Errors found", 1);

            echo $logMessage . "\n";
        }

        /*
         * :: OMEKA BEGIN ::
         * If json file has created then create collection on omeka portal
         */
        if ($this->syncOmeka) {
            try {

                /* Before creating Collection check if already exist */
                $collection = null;
                if (file_exists($inputOmekaInfoPath)) {
                    $str = file_get_contents($inputOmekaInfoPath);
                    $json = json_decode($str, false);

                    if (isset($json->collection->id)) {

                        $collection = $this->omekaAPI->getCollection($json->collection->id);

                        if (isset($collection->message)) {
                            $logMessage = $collection->message;
                            if ($opObject !== null) $opObject->wsLogMessage('<font color="#8b0000">' . $logMessage . '</font>');
                            if ($opObject !== null) $opObject->wsUpdateStatistics($operation_uuid, "Errors found", 1);
                            echo $logMessage . "\n";
                            $collection = null;
                        } else {

                            if (isset($objectJSON->metadata->signature) && trim($objectJSON->metadata->signature) !== '') {
                                if (isset($objectJSON->metadata->code) && trim($objectJSON->metadata->code) !== '') {
                                    $collection->title = $objectJSON->metadata->signature . '-' . $objectJSON->metadata->code;
                                } else {
                                    $collection->title = $objectJSON->metadata->signature;
                                }
                            } else {
                                $collection->title = $objectJSON->metadata->title;
                            }

                            /*
                             * I have to ask if delete all items of the collections and if I do so for all collections already present
                             * */
                            if ($collection->element_texts[3]->text === $objectJSON->metadata->recordid) {

                                /*
                                 * I have to update collection
                                 * */
                                if (!$dryMode) {
                                    $collection = $this->omekaAPI->makeTMBCollection(
                                        $createPublicCollection, false, $objectJSON, $collection->id
                                    );
                                    $this->setElaboratedCollections($this->getElaboratedCollections() + 1);

                                    if (isset($objectJSON->metadata->signature) && trim($objectJSON->metadata->signature) !== '') {
                                        if (isset($objectJSON->metadata->code) && trim($objectJSON->metadata->code) !== '') {
                                            $collection->title = $objectJSON->metadata->signature . '-' . $objectJSON->metadata->code;
                                        } else {
                                            $collection->title = $objectJSON->metadata->signature;
                                        }
                                    } else {
                                        $collection->title = $objectJSON->metadata->title;
                                    }

                                } else {
                                    $message = ">>DRYMODE<< Collection with key: '" . $objectJSON->metadata->recordid . "' should be updated";
                                    if ($opObject !== null) $opObject->wsLogMessage('<font color="#006400">' . $message . '</font>');
                                }

                                if (!$dryMode) {
                                    do {
                                        $items = $this->omekaAPI->getItemsFromCollection($collection->id);
                                        foreach ($items as $item) {
                                            $response = $this->omekaAPI->getHttpClient()->items->delete($item->id);
                                            // echo print_r($response, true);
                                            if ($response->getStatusCode() === 204) {
                                                $message = "Item with id: " . $item->id . ' delete with success!';
                                                if ($opObject !== null) $opObject->wsLogMessage('<font color="#006400">' . $message . '</font>');
                                                if ($opObject !== null) $opObject->wsUpdateStatistics($operation_uuid, "Items deleted", 1);
                                            }
                                        }
                                    } while ($items !== null && is_array($items) && count($items) > 0);
                                } else {
                                    $message = ">>DRYMODE<< Items of collection: " . $collection->id . " should be deleted";
                                    if ($opObject !== null) $opObject->wsLogMessage('<font color="#006400">' . $message . '</font>');
                                }

                            } else {
                                $collection = null;
                            }

                        }
                    }
                }

                if ($collection !== null && isset($collection->id)) {
                    $message = "Collection already exist on omeka portal with recordid: " . $objectJSON->metadata->recordid . ' (Title: ' . $collection->element_texts[0]->text . ')';
                    if ($opObject !== null) $opObject->wsLogMessage('<font color="#006400">' . $message . '</font>');

                    if (isset($objectJSON->metadata->exifData)) {
                        $collection->exifData = $objectJSON->metadata->exifData;
                    }
                    return $collection;
                } else {

                    if (!$dryMode) {
                        $newCollection = $this->omekaAPI->makeTMBCollection(
                            $createPublicCollection, false, $objectJSON
                        );
                        $this->setElaboratedCollections($this->getElaboratedCollections() + 1);

                        if ($newCollection !== null) {

                            if (isset($objectJSON->metadata->signature) && trim($objectJSON->metadata->signature) !== '') {
                                if (isset($objectJSON->metadata->code) && trim($objectJSON->metadata->code) !== '') {
                                    $newCollection->title = $objectJSON->metadata->signature . '-' . $objectJSON->metadata->code;
                                } else {
                                    $newCollection->title = $objectJSON->metadata->signature;
                                }
                            } else {
                                $newCollection->title = $objectJSON->metadata->title;
                            }

                            $message = "Collection created on omeka portal with id: " . $newCollection->id . ' (Title: ' . $newCollection->title/*$newCollection->element_texts[0]->text*/ . ')';
                            if ($opObject !== null) $opObject->wsLogMessage('<font color="#006400">' . $message . '</font>');

                            $record = new Log(null);
                            $record->operation_uuid = $operation_uuid;
                            $record->operation = 'Import';
                            $record->description = $message;

                            if ($opObject !== null) $opObject->wsUpdateStatistics($operation_uuid, "Collections created", 1);

                            $resultInfo = callApi_createLog($record);
                            if (!strpos($resultInfo, 'cURL Error')) {
                                /*$resultInfo = json_decode($resultInfo);
                                $Server->wsSend($clientID, $resultInfo->message);*/
                            } else {
                                if ($opObject !== null) $opObject->wsLogMessage($resultInfo);
                            }

                        } else {
                            if ($opObject !== null) $opObject->wsLogMessage('<font color="#8b0000">' . 'Collection not created!' . '</font>');
                            if ($opObject !== null) $opObject->wsUpdateStatistics($operation_uuid, "Errors found", 1);
                        }
                    } else {
                        $message = ">>DRYMODE<< New collection with key '" . $objectJSON->metadata->recordid . "' and title '" . $objectJSON->metadata->title . "' should be created";
                        if ($opObject !== null) $opObject->wsLogMessage('<font color="#006400">' . $message . '</font>');
                    }

                    if (($newCollection !== null) && (isset($objectJSON->metadata->exifData))) {
                        $newCollection->exifData = $objectJSON->metadata->exifData;
                    }
                    return $newCollection;
                }
            } catch (Exception $e) {
                $logMessage = $e->getMessage();
                if ($opObject !== null) $opObject->wsLogMessage('<font color="#8b0000">' . $logMessage . '</font>');
                if ($opObject !== null) $opObject->wsUpdateStatistics($operation_uuid, "Errors found", 1);
                echo $logMessage . "\n";
                return null;
            }
        }
        /*
         * :: OMEKA END ::
         */
    }

    /**
     * ==================================================
     * tcProcessor
     * ==================================================
     *
     * Method used to process and create tc.json
     *
     * @param object $opObject
     * @param string $operation_uuid
     * @param string $mainFolder
     * @param string $inputFolder
     * @param string $outputFolder
     * @param string $inputMetaDataTCFilePath
     * @param string $collectionTitle
     * @param string $collectionExifData
     * @param integer $collectionId
     * @param boolean $dryMode
     * @param boolean $createOriNameFromID if original name can be get from id
     * @param string $itemTitleField if not specified title of item is collection title + "-" + nome
     * @param boolean $createPublicItems
     * @param object $tcObject
     */
    public function tcProcessor(
        $opObject, $operation_uuid,
        $mainFolder, $inputFolder, $outputFolder,
        $inputMetaDataTCFilePath, $collectionTitle, $collectionId,
        $collectionExifData = "", $dryMode = false, $createOriNameFromID = true, $itemTitleField = '', $createPublicItems = false, $tcObject = null
    )
    {

        $this->php_verbose_mode = true;

        if ($this->php_verbose_mode) {
            error_log("opObject: " . print_r($opObject, true));
            error_log("operation_uuid: " . $operation_uuid);
            error_log("mainFolder: " . $mainFolder);
            error_log("inputFolder: " . $inputFolder);
            error_log("outputFolder: " . $outputFolder);
            error_log("inputMetaDataTCFilePath: " . $inputMetaDataTCFilePath);
            error_log("collectionTitle: " . $collectionTitle);
            error_log("collectionId: " . $collectionId);
        }

        if (isset($tcObject)) {
            $activeObjectJSON = $tcObject;
            $outputFileName = $tcObject->metadata->id;
        } else {

            $excelReader = 'Excel2007';
            if (file_exists($inputMetaDataTCFilePath . '.xlsx')) {
                $inputMetaDataTCFilePath = $inputMetaDataTCFilePath . '.xlsx';
            } else if (file_exists($inputMetaDataTCFilePath . '.xls')) {
                $inputMetaDataTCFilePath = $inputMetaDataTCFilePath . '.xls';
                $excelReader = 'Excel5';
            }

            if ($this->php_verbose_mode) {
                error_log("tcProcessor TMBProcessor method");
                error_log("File to load: " . $inputMetaDataTCFilePath);
            }

            try {

                $errors = array();

                if (!$dryMode) {
                    // Building output file path
                    $outputErrorFilePath = $outputFolder . "/errors.json";
                    // Opening in write mode output file pointer
                    $errorFilePointer = fopen($outputErrorFilePath, 'w');
                }

                $cacheMethod = PHPExcel_CachedObjectStorageFactory:: cache_to_phpTemp;
                $cacheSettings = array(' memoryCacheSize ' => '8MB');
                PHPExcel_Settings::setCacheStorageMethod($cacheMethod, $cacheSettings);

                $objReader = PHPExcel_IOFactory::createReader($excelReader);
                $objReader->setReadDataOnly(true);

                if (TC_METADATAWORKSHEET !== '') {
                    $objReader->setLoadSheetsOnly(array(TC_METADATAWORKSHEET));
                }
                $objectPHPExcel = $objReader->load($inputMetaDataTCFilePath);

                if ($this->php_verbose_mode) {
                    error_log("Building INFO PHPExcel object: OK");
                }

                // Turning PHPExcel object into array
                $arrayPHPExcel = $objectPHPExcel->getActiveSheet()->toArray();
                if ($this->php_verbose_mode) {
                    error_log("Turning PHPExcel object into array: OK");
                }

                if ($this->php_verbose_mode) {
                    error_log("HighestColumn: " . $objectPHPExcel->getActiveSheet()->getHighestColumn());
                }
                if ($this->php_verbose_mode) {
                    error_log("HighestRow: " . $objectPHPExcel->getActiveSheet()->getHighestRow());
                }

                // Use array filter to delete blank columns
                $arrayPHPExcel = array_filter(array_map('array_filter', $arrayPHPExcel));
                if ($this->php_verbose_mode) {
                    error_log("Use array filter to delete blank columns: OK");
                }

                $columnIdentifierArray = $arrayPHPExcel[0];

                // Number of columns to consider is given by title colums
                $columns_number = count($columnIdentifierArray);

                // Building output file path for saving time consumption
                $outputTimesFilePath = $mainFolder . "/times.txt";
                // Opening in write mode output file pointer
                $timesFilePointer = fopen($outputTimesFilePath, 'a+');

                if ($this->syncOmeka) {
                    $itemTypeId = $this->omekaAPI->getItemTypeId(
                        "TMB Book Page"
                    );
                }

                // Consider rows from 1 (row 0 have identifiers)
                for ($row_idx = 1; $row_idx < $objectPHPExcel->getActiveSheet()->getHighestRow(); $row_idx++) {

                    $outputFileName = $row_idx;

                    // Building template JSON stdClass object
                    $activeObjectJSON = new stdClass();
                    $activeObjectJSON->metadata = new stdClass();

                    for ($col_idx = 0; $col_idx < $columns_number; $col_idx++) {

                        if ($col_idx < count($columnIdentifierArray)) {
                            $titleColumn = trim($columnIdentifierArray[$col_idx]);
                            if (isset($arrayPHPExcel[$row_idx][$col_idx]) && trim($arrayPHPExcel[$row_idx][$col_idx]) !== '') {
                                $dataColumn = $arrayPHPExcel[$row_idx][$col_idx];
                            } else {
                                $dataColumn = "";
                            }

                            if ($titleColumn === 'id') {
                                $activeObjectJSON->{$titleColumn} = $dataColumn;
                            } else if (in_array($titleColumn, TC_FIELDS_ARRAY)) {
                                $activeObjectJSON->metadata->{$titleColumn} = $dataColumn;
                            } else {
                                echo "\t!INDEX: \"" . $titleColumn . "\" NOT SUPPORTED!\n";
                            }
                        }

                    }

                    if (($createOriNameFromID) && ($activeObjectJSON->id !== '')) {
                        $activeObjectJSON->metadata->originalFileName = str_pad($activeObjectJSON->id, 4, "0", STR_PAD_LEFT) . ".tif";
                    }

                    if ($activeObjectJSON->id === '') {
                        // If I am on first row then insert blank page
                        if ($row_idx === 1) {
                            $activeObjectJSON->metadata->originalFileName = BLANK_PAGE_NAME . ".tif";
                        } else {
                            $activeObjectJSON->metadata->originalFileName = WHITE_PAPER_NAME . ".tif";
                        }
                    }

                    // At this point check image and if present convert it and then make the json file
                    $singleFile = $inputFolder . '/' . $activeObjectJSON->metadata->originalFileName;

                    // If have a white paper and file not exists I take the default
                    if (($activeObjectJSON->metadata->originalFileName === WHITE_PAPER_NAME . ".tif") && (!file_exists($singleFile))) {
                        $singleFile = "images/" . WHITE_PAPER_NAME . ".tif";
                    }
                    // If have a blank page and file not exists I take the default
                    if (($activeObjectJSON->metadata->originalFileName === BLANK_PAGE_NAME . ".tif") && (!file_exists($singleFile))) {
                        $singleFile = "images/" . BLANK_PAGE_NAME . ".tif";
                    }

                    if ($this->php_verbose_mode) {
                        error_log("Analyzing file " . print_r($singleFile, true) . "...");
                    }

                    fwrite($timesFilePointer, "\n>> Processing " . $singleFile . " <<\n");

                    $partialExecutionTimeStart = microtime(true);

                    $imgConversionReturnObj = $this->imageProcessor(
                        null,
                        $singleFile,
                        $outputFolder . '/' . $outputFileName . '.jpg',
                        $dryMode
                    );

                    if ($dryMode) {
                        $message = ">>DRYMODE<< Image " . $singleFile . " should be converted in " . $outputFolder . '/' . $outputFileName . '.jpg';
                        if ($opObject !== null) $opObject->wsLogMessage('<font color="#006400">' . $message . '</font>');
                    }
                    $partialExecutionTimeEnd = microtime(true);
                    $executionTime = ($partialExecutionTimeEnd - $partialExecutionTimeStart) / 60;
                    fwrite($timesFilePointer, "\nConverting image " . $singleFile . " execution time - | " . $executionTime . " | - Minutes\n");

                    // Write the file, only if image is present and image has been converted
                    if ($imgConversionReturnObj->returnStatus === 'no') {

                        $errorObjectJSON = new stdClass();
                        $errorObjectJSON->errorType = 'File conversion error';
                        $errorObjectJSON->errorDescription = $imgConversionReturnObj->returnDescription;
                        $errors [] = $errorObjectJSON;

                        if ($opObject !== null) $opObject->wsLogMessage('<font color="#8b0000">' . 'File conversion error: ' . $imgConversionReturnObj->returnDescription . '</font>');
                        if ($opObject !== null) $opObject->wsUpdateStatistics($operation_uuid, "Errors found", 1);
                        if ($opObject !== null) $opObject->wsAddError($errorObjectJSON->errorType, $errorObjectJSON->errorDescription);

                        $record = new Log(null);
                        $record->operation_uuid = $operation_uuid;
                        $record->operation = 'Import';
                        $record->description = 'File conversion error: ' . $imgConversionReturnObj->returnDescription;

                        $resultInfo = callApi_createLog($record);
                        if (!strpos($resultInfo, 'cURL Error')) {
                        } else {
                            if ($opObject !== null) $opObject->wsLogMessage($resultInfo);
                        }

                    } else {

                        $activeObjectJSON->{'manifest_id'} = Uuid::uuid4()->toString();
                        // Url originale del file - in questo caso va composto in base a dove vengono caricate le immagini su omeka
                        // $activeObjectJSON->metadata->{'sourceUrl'} = "";
                        $activeObjectJSON->metadata->{'filename'} = $outputFileName . '.jpg';
                        $activeObjectJSON->metadata->{'fileMD5'} = md5_file($outputFolder . '/' . $outputFileName . '.jpg');

                        $activeObjectJSON->{'width'} = $imgConversionReturnObj->width;
                        $activeObjectJSON->{'height'} = $imgConversionReturnObj->height;

                        // Dati tecnici dell'immagine
                        if (trim($collectionExifData) !== '') {
                            if (trim($imgConversionReturnObj->exifData) !== '') {
                                $collectionExifData = $collectionExifData . ";" . trim($imgConversionReturnObj->exifData);
                            }
                            $activeObjectJSON->metadata->{'exifData'} = $collectionExifData;
                        } else {
                            $activeObjectJSON->metadata->{'exifData'} = $imgConversionReturnObj->exifData;
                        }

                        if (!$dryMode) {

                            try {

                                // Building output file path
                                $outputMetadataTCFilePath = $outputFolder . "/" . $outputFileName . ".json";
                                // Opening in write mode output file pointer
                                $filePointer = fopen($outputMetadataTCFilePath, 'w');
                                // Encode JSON into file
                                fwrite($filePointer, json_encode($activeObjectJSON));

                                $logMessage = "File " . $outputMetadataTCFilePath . " created!";
                                if ($this->php_verbose_mode) {
                                    error_log($logMessage);
                                }

                                if ($opObject !== null) $opObject->wsLogMessage('<font color="#006400">' . $logMessage . '</font>');
                                $record = new Log(null);
                                $record->operation_uuid = $operation_uuid;
                                $record->operation = 'Import';
                                $record->description = $logMessage;

                                $resultInfo = callApi_createLog($record);
                                if (!strpos($resultInfo, 'cURL Error')) {
                                    /*$resultInfo = json_decode($resultInfo);
                                    $Server->wsSend($clientID, $resultInfo->message);*/
                                } else {
                                    if ($opObject !== null) $opObject->wsLogMessage($resultInfo);
                                }

                                // Closing previously opened file pointer
                                fclose($filePointer);

                            } catch (Exception $e) {

                                $logMessage = $e->getMessage();
                                if ($opObject !== null) $opObject->wsLogMessage('<font color="#8b0000">' . $logMessage . '</font>');
                                if ($opObject !== null) $opObject->wsUpdateStatistics($operation_uuid, "Errors found", 1);

                                echo $logMessage . "\n";

                            }
                        }

                        if ($this->syncOmeka) {

                            $this->omekaCreateItemAndUploadImage(
                                $opObject,
                                $operation_uuid,
                                $itemTypeId,
                                $outputFolder,
                                $outputFileName,
                                $collectionTitle,
                                $collectionId,
                                $createPublicItems,
                                $activeObjectJSON,
                                $itemTitleField,
                                $timesFilePointer,
                                $dryMode
                            );

                        }

                    }
                }

            } catch (Exception $e) {

                $logMessage = $e->getMessage();
                if ($opObject !== null) $opObject->wsLogMessage('<font color="#8b0000">' . $logMessage . '</font>');
                if ($opObject !== null) $opObject->wsUpdateStatistics($operation_uuid, "Errors found", 1);

                echo $logMessage . "\n";

            }

        }

        if ($this->php_verbose_mode) {
            error_log("Composed tc json: " . print_r($activeObjectJSON, true));
        }

        // :: OMEKA BEGIN ::
        // Get item type id of "TMB Book Page", then make an omeka item of this type
        // and once new item is created attach the image to item
        if (isset($tcObject) && ($this->syncOmeka)) {

            $itemTypeId = $this->omekaAPI->getItemTypeId(
                "TMB Book Page"
            );

            $this->omekaCreateItemAndUploadImage(
                $opObject,
                $operation_uuid,
                $itemTypeId,
                $outputFolder,
                $outputFileName,
                $collectionTitle,
                $collectionId,
                $createPublicItems,
                $activeObjectJSON,
                $itemTitleField,
                $timesFilePointer,
                $dryMode
            );

        }
        // :: OMEKA END ::

        $data = array('errors' => $errors);
        fwrite($errorFilePointer, json_encode($data, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));

        // Closing previously opened file pointer
        fclose($errorFilePointer);

        if (isset($objectPHPExcel)) {
            $objectPHPExcel->disconnectWorksheets();
            unset($objectPHPExcel);
        }

    }

    /**
     * @param $opObject
     * @param $operation_uuid
     * @param $itemTypeId
     * @param $outputFolder
     * @param $outputFileName
     * @param $collectionTitle
     * @param $collectionId
     * @param $createPublicItems
     * @param $activeObjectJSON
     * @param $itemTitleField
     * @param $timesFilePointer
     * @param $dryMode
     */
    private function omekaCreateItemAndUploadImage(
        $opObject,
        $operation_uuid,
        $itemTypeId,
        $outputFolder,
        $outputFileName,
        $collectionTitle,
        $collectionId,
        $createPublicItems,
        $activeObjectJSON,
        $itemTitleField,
        $timesFilePointer,
        $dryMode
    )
    {

        try {

            if (isset($itemTypeId)) {

                $partialExecutionTimeStart = microtime(true);
                if (!$dryMode) {

                    $newItem = $this->omekaAPI->makeTMBItem(
                        $itemTypeId, $collectionTitle, $collectionId, $createPublicItems, false, $activeObjectJSON, $itemTitleField
                    );
                    $this->setCurrentElaboratedItems($this->getCurrentElaboratedItems() + 1);
                    $this->setElaboratedItems($this->getElaboratedItems() + 1);

                    if ($this->getCurrentElaboratedItems() >= $this->getMaxElaboratedItemsForSendingEmail()) {
                        $this->setCurrentElaboratedItems(0);
                        $secondsFromStart = round(microtime(true), 3) - round($this->getTimeTMBProcessorStarted(), 3);

                        if ($opObject !== null) {
                            $opObject->sendMail(
                                "Status of import on machine " . gethostname(),
                                "<b>Status of import after " . $opObject->formatSecondsToReadableString($secondsFromStart) . ":</b><br>" .
                                "Elaborated collections: " . $this->getElaboratedCollections() . "<br>" .
                                "Elaborated items: " . $this->getElaboratedItems()
                            );
                        }
                    }

                } else {

                    $message = ">>DRYMODE<< New item associated to image '" . $activeObjectJSON->metadata->{'filename'} . "' should be created";
                    if ($opObject !== null) $opObject->wsLogMessage('<font color="#006400">' . $message . '</font>');

                }

                $partialExecutionTimeEnd = microtime(true);
                $executionTime = ($partialExecutionTimeEnd - $partialExecutionTimeStart) / 60;

                if (isset($activeObjectJSON->metadata->filename)) {
                    fwrite($timesFilePointer, "\nCreating omeka item " . $activeObjectJSON->metadata->filename . " execution time - | " . $executionTime . " | - Minutes\n");
                }

                if (isset($newItem) && isset($newItem->id)) {

                    $logMessage = "Item created on omeka portal with id: " . $newItem->id . ' (Title: ' . $newItem->element_texts[0]->text . ')';
                    if ($this->php_verbose_mode) {
                        error_log($logMessage);
                    }

                    if ($opObject !== null) $opObject->wsLogMessage('<font color="#006400">' . $logMessage . '</font>');
                    $record = new Log(null);
                    $record->operation_uuid = $operation_uuid;
                    $record->operation = 'Import';
                    $record->description = $logMessage;

                    if ($opObject !== null) $opObject->wsUpdateStatistics($operation_uuid, "Items created", 1);

                    $resultInfo = callApi_createLog($record);
                    if (!strpos($resultInfo, 'cURL Error')) {
                        /*$resultInfo = json_decode($resultInfo);
                        $Server->wsSend($clientID, $resultInfo->message);*/
                    } else {
                        if ($opObject !== null) $opObject->wsLogMessage($resultInfo);
                    }

                    // Before upload file set source item

                    // Per adesso non va, fai delle verifiche
                    /*
                    $this->omekaAPI->updateTMBItemSource(
                        $newItem->id, 'http://iiif-server.mind-ware.it/omeka/oa/items/' . $newItem->id . '/manifest.json'
                    );
                    */

                    $partialExecutionTimeStart = microtime(true);

                    if ($this->php_verbose_mode) {
                        error_log("Uploading file " . $outputFolder . "/" . $outputFileName . ".jpg associated to omeka item with id " . $newItem->id);
                    }

                    $result = $this->omekaAPI->uploadFile(
                        $newItem->id, $outputFolder . "/" . $outputFileName . ".jpg"
                    );
                    $partialExecutionTimeEnd = microtime(true);

                    if ($result !== null) {

                        $logMessage = "Image " . $outputFolder . "/" . $outputFileName . ".jpg associated to omeka item with id " . $newItem->id . " loaded with success!";
                        if ($opObject !== null) $opObject->wsLogMessage('<font color="#006400">' . $logMessage . '</font>');
                        $record = new Log(null);
                        $record->operation_uuid = $operation_uuid;
                        $record->operation = 'Import';
                        $record->description = $logMessage;

                        if ($opObject !== null) $opObject->wsUpdateStatistics($operation_uuid, "Images uploaded", 1);

                        $resultInfo = callApi_createLog($record);
                        if (!strpos($resultInfo, 'cURL Error')) {
                            /*$resultInfo = json_decode($resultInfo);
                            $Server->wsSend($clientID, $resultInfo->message);*/
                        } else {
                            if ($opObject !== null) $opObject->wsLogMessage($resultInfo);
                        }

                    }
                    $executionTime = ($partialExecutionTimeEnd - $partialExecutionTimeStart) / 60;
                    fwrite($timesFilePointer, "\nUploading file " . $outputFolder . "/" . $outputFileName . ".jpg" . " execution time - | " . $executionTime . " | - Minutes\n");

                } else if (!$dryMode) {

                    if ($opObject !== null) $opObject->wsLogMessage('<font color="#8b0000">' . 'Item not created!' . '</font>');
                    if ($opObject !== null) $opObject->wsUpdateStatistics($operation_uuid, "Errors found", 1);

                }

            }

        } catch (Exception $e) {

            $logMessage = $e->getMessage();
            if ($opObject !== null) $opObject->wsLogMessage('<font color="#8b0000">' . $logMessage . '</font>');
            if ($opObject !== null) $opObject->wsUpdateStatistics($operation_uuid, "Errors found", 1);

            echo $logMessage . "\n";

        }

    }

}