<?php
/**
 * Created by PhpStorm.
 * User: fbeninat
 * Date: 25/07/17
 * Time: 09:34
 */

require_once ('omekaHttpClient.php');

// Include database and object files for using SQLite
include_once __DIR__ . '/../api/config/database.php';
include_once __DIR__ . '/../api/objects/omekaConf.php';

/**
 * ==================================================
 * OmekaAPIWrapper
 * ==================================================
 *
 * Class used to wrapper and manage omeka API
 *
 */
class OmekaAPIWrapper
{

    /** @var boolean verbose mode, default is false */
    private $verboseMode = false;

    /** @var OmekaHttpClient http client created for http requests to omeka api. */
    private $httpClient = null;

    /**
     * @return OmekaHttpClient
     */
    public function getHttpClient()
    {
        return $this->httpClient;
    }

    /**
     * Constructor
     *
     * @param boolean $useConfigOnDb
     */
    public function __construct($useConfigOnDb = false)
    {

        if (!$useConfigOnDb) {

            $omekaconfigs = include('omekaConfig.php');

        } else {
            /* Load data from sqlite */
            // Instantiate database and product object
            $database = new Database();
            $db = $database->getConnection();

            // Initialize object
            $omekaConf = new omekaConf($db->PDO);

            $omekaconfigs = array();

            $ret = $omekaConf->read();
            while ($row = $ret->fetch(SQLITE3_ASSOC)) {

                $omekaconfigs = array(
                    "apiBaseUrl" => $row['apiBaseUrl'],
                    "apiToken" => $row['apiToken'],
                    "debug" => $row['debug'],
                );

                break;

            }
        }

        $this->httpClient = new OmekaHttpClient($omekaconfigs['apiBaseUrl']);
        $this->httpClient->setKey($omekaconfigs['apiToken']);
        $this->httpClient->setDebug($omekaconfigs['debug']);
    }

    /**
     * ==================================================
     * getVerboseMode
     * ==================================================
     *
     * Method used to get verbose mode state
     *
     * @return boolean Return the verbose mode.
     */
    public function getVerboseMode() {
        return $this->verboseMode;
    }

    /**
     * ==================================================
     * setVerboseMode
     * ==================================================
     *
     * Method used to set verbose mode state
     *
     * @param boolean $flag verbose mode state flag.
     */
    public function setVerboseMode($flag) {
        $this->verboseMode = $flag;
    }

    /**
     * ==================================================
     * getElementsSetId
     * ==================================================
     *
     * Method used to get id of an omeka elements set
     *
     * @param string $name name of an elements set
     *
     * @return integer|null elements set id
     */
    public function getElementsSetId(
        $name
    ) {

        $parameters = [];
        $parameters['name'] = $name;

        $response = $this->httpClient->element_sets->get($parameters);
        //echo print_r(Zend\Json\Json::decode($response->getBody()), true);
        $response = Zend\Json\Json::decode($response->getBody())[0];

        if (isset($response) && isset($response->id)) {
            return $response->id;
        } else {
            return null;
        }

    }

    /**
     * ==================================================
     * getElements
     * ==================================================
     *
     * Method used to get elements of an omeka elements set
     *
     * @param integer $id id of an omeka elements set
     * @param string $filter the variabile can have the value 'element_set' (default) or 'item_type'
     *
     * @return array|null elements set
     */
    public function getElements(
        $id, $filter = 'element_set'
    ) {

        $parameters = [];
        $parameters[$filter] = $id;
        $response = $this->httpClient->elements->get($parameters);
        $elements = Zend\Json\Json::decode($response->getBody());

        if (isset($elements) && is_array($elements)) {
            return $elements;
        } else {
            return null;
        }

    }

    /**
     * ==================================================
     * getItemTypeId
     * ==================================================
     *
     * Method used to get id of an omeka item type
     *
     * @param string $name name of an omeka item type
     *
     * @return integer|null elements set id
     */
    public function getItemTypeId(
        $name
    ) {

        $parameters = [];
        $parameters['name'] = $name;
        $response = $this->httpClient->item_types->get($parameters);
        //echo print_r(Zend\Json\Json::decode($response->getBody()), true);
        $response = Zend\Json\Json::decode($response->getBody())[0];

        if (isset($response) && isset($response->id)) {
            return $response->id;
        } else {
            return null;
        }

    }

    /**
     * ==================================================
     * getItems
     * ==================================================
     *
     * Method used to get items from an adding date (ISO 8601 format) or a modifying date (ISO 8601 format)
     *
     * @param string $added_since date in which items were added.
     * @param string $modified_since date in which items were modified.
     *
     * @return array|null items
     */
    public function getItems(
        $added_since = null, $modified_since = null
    ) {

        if (isset($added_since) || isset($modified_since)) {
            $parameters = [];
            if (isset($added_since)) {
                $parameters['added_since'] = $added_since;
            }
            if (isset($modified_since)) {
                $parameters['modified_since'] = $modified_since;
            }

            $response = $this->httpClient->items->get($parameters);
            $items = Zend\Json\Json::decode($response->getBody());

            if (isset($items)) {
                return $items;
            } else {
                return null;
            }
        }

    }

    /**
     * ==================================================
     * getItemsFromCollection
     * ==================================================
     *
     * Method used to get items a collection id
     *
     * @param integer $collection_id
     * @return array|null items
     */
    public function getItemsFromCollection(
        $collection_id
    ) {

        if (isset($collection_id)) {
            $parameters = [];
            $parameters['collection'] = $collection_id;

            $response = $this->httpClient->items->get($parameters);
            $items = Zend\Json\Json::decode($response->getBody());

            if (is_array($items)) {
                return $items;
            } else {
                return null;
            }
        } else {
            return null;
        }

    }

    /**
     * ==================================================
     * getItem
     * ==================================================
     *
     * Method used to get item from item id
     *
     * @param integer $id
     * @return object|null
     */
    public function getItem(
        $id
    ) {

        if (isset($id)) {

            $response = $this->httpClient->items->get($id);
            $item = Zend\Json\Json::decode($response->getBody());

            if (isset($item)) {
                return $item;
            } else {
                return null;
            }
        }

    }

    /**
     * ==================================================
     * deleteCollectionItems
     * ==================================================
     *
     * Method used to delete all items of a collection
     *
     * @param integer $collection_id id of a collection
     * @return bool
     */
    public function deleteCollectionItems(
        $collection_id
    ) {
        if (isset($collection_id)) {
            $parameters = [];
            $parameters['collection'] = $collection_id;

            $response = $this->httpClient->items->get($parameters);
            $items = Zend\Json\Json::decode($response->getBody());

            if (isset($items)) {
                echo "Founded " . count($items) . " items to delete\n";
                foreach ($items as $item) {
                    echo "Deleting item with id " . $item->id . "...\n";
                    $this->httpClient->items->delete($item->id);
                }
            } else {
                return null;
            }
        } else {
            return null;
        }
    }

    /**
     * ==================================================
     * getCollections
     * ==================================================
     *
     * Method used to get collections from an adding date (ISO 8601 format) or a modifying date (ISO 8601 format)
     *
     * @param string $added_since date in which collections were added.
     * @param string $modified_since date in which collections were modified.
     *
     * @return array|null collections
     */
    public function getCollections(
        $added_since = null, $modified_since = null
    ) {

        if (isset($added_since) || isset($modified_since)) {
            $parameters = [];
            if (isset($added_since)) {
                $parameters['added_since'] = $added_since;
            }
            if (isset($modified_since)) {
                $parameters['modified_since'] = $modified_since;
            }

            $response = $this->httpClient->collections->get($parameters);
            $collections = Zend\Json\Json::decode($response->getBody());

            if (isset($collections)) {
                return $collections;
            } else {
                return null;
            }
        }

    }

    /**
     * ==================================================
     * getCollection
     * ==================================================
     *
     * Method used to get collection from collection id
     *
     * @param integer $id
     * @return object|null
     */
    public function getCollection(
        $id
    ) {

        if (isset($id)) {

            $response = $this->httpClient->collections->get($id);
            $collection = Zend\Json\Json::decode($response->getBody());

            if (isset($collection)) {
                return $collection;
            } else {
                return null;
            }
        }

    }

    /**
     * ==================================================
     * getLastDaysAddedItems
     * ==================================================
     *
     * Method used to get added items of last days
     *
     * @param integer $days number of days
     *
     * @return array|null items
     */
    public function getLastDaysAddedItems(
        $days
    ) {

        if (isset($days) && $days > 0) {
            $parameters = [];

            // From now I calculate ISO 8601 date of last $days
            $added_since = new DateTime('NOW');
            $added_since = $added_since->modify( '-' . $days . ($days === 1 ? 'day' : 'days') );
            $added_since = $added_since->format(DateTime::ATOM);

            $parameters['added_since'] = $added_since;

            $response = $this->httpClient->items->get($parameters);
            $items = Zend\Json\Json::decode($response->getBody());

            if (isset($items)) {
                return $items;
            } else {
                return null;
            }
        } else {
            return null;
        }

    }

    /**
     * ==================================================
     * getLastDaysModifiedItems
     * ==================================================
     *
     * Method used to get modified items of last days
     *
     * @param integer $days number of days
     *
     * @return array|null items
     */
    public function getLastDaysModifiedItems(
        $days
    ) {

        if (isset($days) && $days > 0) {
            $parameters = [];

            // From now I calculate ISO 8601 date of last $days
            $modified_since = new DateTime('NOW');
            $modified_since = $modified_since->modify( '-' . $days . ($days === 1 ? 'day' : 'days') );
            $modified_since = $modified_since->format(DateTime::ATOM);

            $parameters['modified_since'] = $modified_since;

            $response = $this->httpClient->items->get($parameters);
            $items = Zend\Json\Json::decode($response->getBody());

            if (isset($items)) {
                return $items;
            } else {
                return null;
            }
        } else {
            return null;
        }

    }

    /**
     * ==================================================
     * itemFilesDownload
     * ==================================================
     *
     * Method used to download related files of an item
     *
     * @param integer $item_id id of the omeka item
     * @param string $download_type type of download ['all', 'files', 'manifest']
     *
     * @return array|null items
     */
    public function itemFilesDownload(
        $item_id, $download_type
    ) {

        if (isset($item_id) && ($item_id >=0) ) {

            $parameters = [];
            $param = new stdClass();

            switch ($download_type) {
                case 'all':
                    $param->function = 'download-related-all';
                    break;
                case 'files':
                    $param->function = 'download-related-files';
                    break;
                case 'manifest':
                    $param->function = 'download-related-manifest';
                    break;
            }

            $param->debug = false;
            $parameters [] = $param;
            $response = $this->httpClient->oa->get($item_id, $parameters);

            $files = Zend\Json\Json::decode($response->getBody());

            if (is_array($files)) {
                return $files;
            } else {
                return null;
            }

        } else {
            return null;
        }

    }

    /**
     * ==================================================
     * collectionFilesDownload
     * ==================================================
     *
     * Method used to download related files of a collection
     *
     * @param integer $collection_id id of the omeka collection
     * @param string $download_type type of download ['manifest']
     *
     * @return array|null items
     */
    public function collectionFilesDownload(
        $collection_id, $download_type
    ) {

        if (isset($collection_id) && ($collection_id >=0) ) {

            $parameters = [];
            $param = new stdClass();

            switch ($download_type) {
                case 'manifest':
                case 'all':
                case 'files':
                    $param->function = 'download-related-manifest';
                    break;
            }

            $param->debug = false;
            $param->type = 'Collection';
            $parameters [] = $param;
            $response = $this->httpClient->oa->get($collection_id, $parameters);

            $files = Zend\Json\Json::decode($response->getBody());

            if (is_array($files)) {
                return $files;
            } else {
                return null;
            }

        } else {
            return null;
        }

    }

    /**
     * ==================================================
     * uploadFile
     * ==================================================
     *
     * Method used to get id of an omeka item type
     *
     * @param integer $itemId id of the item that the image have to be attached
     * @param string $filePath full path with file name and extension of the file
     *
     * @return integer|null file id
     */
    public function uploadFile(
        $itemId, $filePath
    ) {

        // Attach image to item
        $response = $this->httpClient->files->post($filePath, '{"item":{"id":' . $itemId . '}}');
        $response = Zend\Json\Json::decode($response->getBody());

        if (isset($response) && isset($response->id)) {
            if ($this->getVerboseMode()) { echo "File " . $filePath . " uploaded with success!\nInfo: " . $response->type_os . ".\n"; }
            return $response->id;
        } else {
            return null;
        }

    }

    /**
     * ==================================================
     * seekAndFound
     * ==================================================
     *
     * Method used to get id of an omeka item type
     *
     * @param string $type type of the omeka record (item or collection)
     * @param string $fieldname name of the search field
     * @param string $fieldvalue value of the search field
     *
     * @return object|null omeka object
     */
    /*
    public function seekAndFound($type, $fieldname, $fieldvalue) {

        if ($this->getVerboseMode()) { echo "Searching omeka collection by " . $fieldname . " with value " . $fieldvalue . "...\n"; }

        $result = null;
        switch ($type) {
            case 'item':
                echo "not implemented";
                break;
            case 'collection':
                break;
        }
        return $result;
    }
    */

    /**
     * ==================================================
     * makeTMBCollection
     * ==================================================
     *
     * Method used to create a new TMB Collection. If $id is passed this
     * method try to edit collection with put request
     *
     * @param boolean $public flag that if true means that item is public
     * @param boolean $featured flag that if true means that item is featured
     * @param object $data object that contains data as defined for TMB collection
     * @param integer $id id of existing TMB collection
     *
     * @throws Exception An exception is called if an elements set or elements not founded
     *
     * @return object|null
     */
    public function makeTMBCollection(
        $public, $featured, $data, $id = null
    ) {

        if ($this->getVerboseMode()) {
            if (isset($id)) {
                echo "Editing omeka collection with id " . $id . "...\n";
            } else {
                echo "Creating omeka collection...\n";
            }
        }

        try {
            $dcElementSetId = $this->getElementsSetId('Dublin Core');
            if ($dcElementSetId === null) { throw new Exception('Dublin Core not found!'); }

            $elements = $this->getElements($dcElementSetId);
            if ($elements === null) { throw new Exception('Dublin Core elements not found!'); }
        } catch (Exception $e) {
            if ($this->getVerboseMode()) { echo "Error on retrieving Dublin Core Id or Dublin Core elements!\n" . $e->getMessage() . "\n"; }
            return null;
        }

        if (is_array($elements)) {
            foreach ($elements as $element) {
                switch ($element->name) {
                    case "Title":
                        $dcTitleId = $element->id;
                        break;
                    case "Identifier":
                        $dcIdentifierId = $element->id;
                        break;

                }
            }
        }

        /*
         * Get elementSetName elements
         * Check in API configuration
         * Maximum number of API results displayed per page
         * (default 50) and adjust for returning all results!!
         */
        try {
            $tmbcElementSetId = $this->getElementsSetId('TMB Collection');
            if ($tmbcElementSetId === null) {
                throw new Exception('TMB Collection not found!');
            }

            $elements = $this->getElements($tmbcElementSetId);
            if ($elements === null) {
                throw new Exception('TMB Collection elements not found!');
            }
        } catch (Exception $e) {
            if ($this->getVerboseMode()) { echo $e->getMessage() . "\n"; }
            return null;
        }

        $parameters = [];
        $parameters['public'] = $public;
        $parameters['featured'] = $featured;
        $parameters['element_texts'] = [];

        // As first thing You have to set Dublin Core Title
        $text_element = new stdClass();
        $text_element->html = false;

        if (isset($data->metadata->signature) && trim($data->metadata->signature) !== '') {
            if (isset($data->metadata->code) && trim($data->metadata->code) !== '') {
                $text_element->text = $data->metadata->title . ' - ' . $data->metadata->signature . ' - ' . $data->metadata->code;
            } else {
                $text_element->text = $data->metadata->title . ' - ' . $data->metadata->signature;
            }
        } else {
            $text_element->text = $data->metadata->title;
        }

        // $text_element->text = $data->name;
        $new_element = new stdClass();
        $new_element->id = $dcTitleId;
        $text_element->element = $new_element;
        $parameters['element_texts'][] = $text_element;
        // As first thing You have to set Dublin Core Identifier
        $text_element = new stdClass();
        $text_element->html = false;
        $text_element->text = $data->id;
        $new_element = new stdClass();
        $new_element->id = $dcIdentifierId;
        $text_element->element = $new_element;
        $parameters['element_texts'][] = $text_element;

        /*
         * Cycle TMB Collection elements and match with data $data->metadata
         * */
        foreach ($elements as $element) {
            $metadata_name = $element->name;
            if (isset($data->metadata->$metadata_name) && trim($data->metadata->$metadata_name) !== '') {
                $text_element = new stdClass();
                $text_element->html = false;
                $text_element->text = $data->metadata->$metadata_name;
                $new_element = new stdClass();
                $new_element->id = $element->id;
                $text_element->element = $new_element;
                $parameters['element_texts'][] = $text_element;
            }
        }

        $jsonObject = Zend\Json\Json::encode($parameters, true);
        if ($this->getVerboseMode()) { echo $jsonObject . "\n"; }

        if (isset($id)) {
            $response = $this->httpClient->collections->put($id, $jsonObject);
        } else {
            $response = $this->httpClient->collections->post($jsonObject);
        }

        $newCollection = Zend\Json\Json::decode($response->getBody());

        if (($newCollection !== null) && isset($newCollection->id)) {

            if ($this->getVerboseMode()) {
                if (isset($id)) {
                    echo "Omeka collection edited with id: " . $newCollection->id . "\n";
                } else {
                    echo "Omeka collection created with id: " . $newCollection->id . "\n";
                }
            }
            return $newCollection;
        } else {
            return null;
        }

    }

    /**
     * ==================================================
     * makeTMBItem
     * ==================================================
     *
     * Method used to create a new TMB Item
     *
     * @param integer $element_type_id id of an omeka element type
     * @param string $collection_title title of an omeka collection
     * @param integer $collection_id id of an omeka collection
     * @param boolean $public flag that if true means that item is public
     * @param boolean $featured flag that if true means that item is featured
     * @param object $data object that contains data as defined for TMB item of a particular $element_type_id
     * @param string $titleField if not specified title of item is collection title + "-" + nome immagine
     *
     * @throws Exception An exception is called if an elements set or elements not founded
     *
     * @return object|null
     */
    public function makeTMBItem(
        $element_type_id,  $collection_title, $collection_id, $public, $featured, $data, $titleField = ''
    ) {

        if ($this->getVerboseMode()) {
            echo "Creating omeka item...\n";
            /*
            echo "collection_title: " . $collection_title . "\n";
            echo "collection_id: " . $collection_id . "\n";
            echo "public: " . $public . "\n";
            echo "featured: " . $featured . "\n";
            echo "data: " . print_r($data, true) . "\n";
            echo "titleField: " . $titleField . "\n";
            */
        }

        try {
            $dcElementSetId = $this->getElementsSetId('Dublin Core');
            if ($dcElementSetId === null) { throw new Exception('Dublin Core not found!'); }

            $elements = $this->getElements($dcElementSetId);
            if ($elements === null) { throw new Exception('Dublin Core elements not found!'); }
        } catch (Exception $e) {
            if ($this->getVerboseMode()) { echo $e->getMessage() . "\n"; }
            return null;
        }

        if (is_array($elements)) {
            foreach ($elements as $element) {
                switch ($element->name) {
                    case "Title":
                        $dcTitleId = $element->id;
                        break;
                    case "Identifier":
                        $dcIdentifierId = $element->id;
                        break;

                }
            }
        }

        try {
            $elements = $this->getElements($element_type_id, 'item_type');
            if (($elements === null) || (is_array($elements) && count($elements) === 0)) { throw new Exception('Elements of type ' . $element_type_id . ' not found!'); }
        } catch (Exception $e) {
            if ($this->getVerboseMode()) { echo $e->getMessage() . "\n"; }
            return null;
        }

        $parameters = [];
        $parameters['public'] = $public;
        $parameters['featured'] = $featured;

        $item_type = new stdClass();
        $item_type->id = $element_type_id;
        $parameters['item_type'] = $item_type;

        $collection = new stdClass();
        $collection->id = $collection_id;
        $parameters['collection'] = $collection;

        $parameters['element_texts'] = [];

        // As first thing You have to set Dublin Core Title
        $text_element = new stdClass();
        $text_element->html = false;
        $text_element->text = "";
        if (trim($titleField) !== '') {
            if (isset($data->metadata->{$titleField})) {
                $text_element->text = $data->metadata->{$titleField};
            }
        } else {
            if (isset($data->metadata->filename)) {
                $text_element->text = $collection_title . "-" . basename($data->metadata->filename, '.jpg');
            }
        }

        if (trim($text_element->text) !== '') {
            $new_element = new stdClass();
            $new_element->id = $dcTitleId;
            $text_element->element = $new_element;
            $parameters['element_texts'][] = $text_element;
        }

        // As first thing You have to set Dublin Core Identifier
        if (isset($data->manifest_id)) {
            $text_element = new stdClass();
            $text_element->html = false;
            $text_element->text = $data->manifest_id;
            $new_element = new stdClass();
            $new_element->id = $dcIdentifierId;
            $text_element->element = $new_element;
            $parameters['element_texts'][] = $text_element;
        }

        foreach ($elements as $element) {
            $metadata_name = $element->name;
            if (isset($data->metadata->$metadata_name) && trim($data->metadata->$metadata_name) !== '') {
                $text_element = new stdClass();
                $text_element->html = false;
                $text_element->text = $data->metadata->$metadata_name;
                $new_element = new stdClass();
                $new_element->id = $element->id;
                $text_element->element = $new_element;
                $parameters['element_texts'][] = $text_element;
            }
        }

        $jsonObject = Zend\Json\Json::encode($parameters, true);

        $response = $this->httpClient->items->post($jsonObject);

        $newItem = Zend\Json\Json::decode($response->getBody());

        if (($newItem!== null) && isset($newItem->id)) {
            if ($this->getVerboseMode()) { echo "Omeka item created with id: " . $newItem->id  . "\n"; }
            return $newItem;
        } else {
            return null;
        }

    }


    /* ATTENTION! THIS NOT WORK AT THIS MOMENT! */

    /**
     * ==================================================
     * updateTMBItemSource
     * ==================================================
     *
     * Method used to update "Dublin Core" source field of a TMB Item
     *
     * @param integer $item_id id of an omeka item
     * @param string $source original url of the item
     *
     * @throws Exception An exception is called if an elements set or elements or item id not founded
     *
     * @return object|null
     */
    /*
    public function updateTMBItemSource(
        $item_id, $source
    ) {

        if ($this->getVerboseMode()) { echo "Updating omeka item with id: " . $item_id . "...\n"; }

        try {
            $dcElementSetId = $this->getElementsSetId('Dublin Core');
            if ($dcElementSetId === null) { throw new Exception('Dublin Core not found!'); }

            $elements = $this->getElements($dcElementSetId);
            if ($elements === null) { throw new Exception('Dublin Core elements not found!'); }
        } catch (Exception $e) {
            if ($this->getVerboseMode()) { echo $e->getMessage() . "\n"; }
            return null;
        }

        if (is_array($elements)) {
            foreach ($elements as $element) {
                switch ($element->name) {
                    case "Source":
                        $dcSourceId = $element->id;
                        break;
                }
            }
        }

        $parameters['element_texts'] = [];

        // set Dublin Core Source
        $text_element = new stdClass();
        $text_element->html = false;
        $text_element->text = $source;
        $new_element = new stdClass();
        $new_element->id = $dcSourceId;
        $text_element->element = $new_element;
        $parameters['element_texts'][] = $text_element;

        $jsonObject = Zend\Json\Json::encode($parameters, true);

        $response = $this->httpClient->items->put($item_id, $jsonObject);

        $newItem = Zend\Json\Json::decode($response->getBody());

        if (($newItem!== null) && isset($newItem->id)) {
            if ($this->getVerboseMode()) { echo "Omeka item with id: " . $newItem->id  . " updated with success!!\n"; }
            return $newItem;
        } else {
            return null;
        }

    }
    */

}